*=============================================================================
* File      : SignUserDemo.gms
* Author    : Wietse Dol (w.dol@wur.nl)
* Version   : 1.0
* Date      : 14-Nov-10 18:29:23
* Changed   : 09-Jan-11 22:07:45
* Changed by: Wietse Dol (w.dol@wur.nl)
* Remarks   :
$ontext
   Sign testing and User testing examples.
   These examples show how one can create readable code for checking data.

   Note that Sign testing and User testing is done the best by using the setglobals see:
   Example 1 and Example 2. The other examples show additional possibilities.

%SignTest% has 6 arguments:
   %Data%        parameter to be checked (including sets/subsets)
   %Tuple%       default: All i.e. checl all elements of %Data%
   %Condition%   StrictlyNegative, Negative, Zero, Positive, StrictlyPositive
   %OutName%     default: %Condition%
   %CorrectName% default: Correct
   %WrongName%   default: Wrong
*- OUTPUT: %OutName%%CorrectName%%Data%      default: %Condition%Correct%Data%
*- OUTPUT: %OutName%%WrongName%%Data%        default: %Condition%Wrong%Data%


%UserTest% has 7 arguments:
   %Data%        parameter to be checked (including sets/subsets)
   %Tuple%       default: All i.e. checl all elements of %Data%
   %Correct%     condition to be correct
   %Wrong%       default: Otherwise i.e. everything that is not %Correct%
   %OutName%     default: empty
   %CorrectName% default: Correct
   %WrongName%   default: Wrong
*- OUTPUT: %OutName%%CorrectName%%Data%      default: Correct%Data%
*- OUTPUT: %OutName%%WrongName%%Data%        default: Wrong%Data%

NB when you do not specify an argument the default is taken:
*!   %SignTest% needs %Data% and %Condition% and
*!   %UserTest% needs %Data% and %Correct%
$offtext
*=============================================================================
*_ The commands and settings
*1. Gtree tags
*! <%GTREECONTROL CLOSEWLOG%>
*! <%GTREECONTROL JUMPLIST "@filenamenoext@.lst",0,0,"","GAMS LST file" %>
*! <%GTREECONTROL JUMPLIST "@pathname@..\..\..\doc\Sign testing in GAMS.doc",0,0,"","Sign testing in GAMS"%>
*! <%GTREECONTROL OPENLSTAFTERCOMPILE%>

*2. Progress display
$setGlobal WLOG no
*switch to yes if you want to use wlog
*switch to no  if you do not want to use wlog

*3. Settings
$include Settings.gms
*=============================================================================
*_ Sets and data
set TreeElements
/
"Chapter1"
   "Section1"
      "SubSection1.1"
         "SubSubSection1.1.1"
         "SubSubSection1.1.2"
      "SubSection1.2"
   "Section2"
      "SubSection2.1"
"Chapter2"
   "Section3"
"Chapter3"
/;

set Years/2000*2010/;

table IO(TreeElements,Years)
                                   2000   2005  2010
  "Chapter1"                        100    110   120
     "Section1"                     -10      0   100
        "SubSection1.1"              30     40    60
           "SubSubSection1.1.1"      10     20    30
           "SubSubSection1.1.2"      20     25    30
        "SubSection1.2"              40     50    40
     "Section2"                      30     20    40
           "SubSection2.1"           30     20    35
  "Chapter2"                        200    110   120
     "Section3"                     100    110   120
  "Chapter3"                       -300   -200   210
;

parameter Upper(Years)
/
  2000 100
  2005 150
  2010 250
/;

parameter Lower(Years)
/
  2000  50
  2005 100
  2010 150
/;

*-=============================================================================
*-=====       SIGN TESTING                                                =====
*-=============================================================================
*! <%GTREE Example 1: Sign testing using globals%>
*- minimal statement to test if all elements of IO are positive:
%Data%        IO(TreeElements,Years)
%Condition%   Positive

%SignTest%
*The test generates the output:
*   set PositiveCorrectIO(TreeElements,Years)
*   set PositiveWrongIO(TreeElements,Years)
%userdisplay% PositiveCorrectIO,PositiveWrongIO;
*=============================================================================
set TestConditions(TreeElements,Years)
/
(Chapter1,Section1,Chapter2,Chapter3) . (2000,2010)
/;

*- instead of testing all elements of IO we test elements that are specified by the %Tuple%
%Data%        IO(TreeElements,Years)
%Tuple%       TestConditions(TreeElements,Years)
%Condition%   StrictlyPositive

%SignTest%
%userdisplay% StrictlyPositiveCorrectIO,StrictlyPositiveWrongIO;
*=============================================================================
*- you can name the output sets of the test
%Data%        IO(TreeElements,Years)
%Tuple%       TestConditions(TreeElements,Years)
%Condition%   Positive
%CorrectName% _OK_
%WrongName%   _Error_

%SignTest%
%userdisplay% Positive_OK_IO,Positive_Error_IO;
*=============================================================================
*- you can name the output sets of the test
%Data%        IO(TreeElements,Years)
%Tuple%       TestConditions(TreeElements,Years)
%Condition%   Positive
%OutName%     Test
%CorrectName% Ok
%WrongName%   Error

%SignTest%
%userdisplay% TestOkIO,TestErrorIO;

*- it is easy to correct the data that is wrong and recheck the data
parameter Errors(TreeElements,Years) 'Data that contains errors';
Errors(TreeElements,Years)$TestErrorIO(TreeElements,Years)=IO(TreeElements,Years);
%userdisplay% Errors;
*- correct the data
IO(TreeElements,Years)$TestErrorIO(TreeElements,Years)=abs(IO(TreeElements,Years));
*-check the data again (there should be no errors now!)
%Data%        IO(TreeElements,Years)
%Tuple%       TestConditions(TreeElements,Years)
%Condition%   Positive
%OutName%     TestAgain
%CorrectName% Ok
%WrongName%   Error

%SignTest%
%userdisplay% TestAgainErrorIO;

*-=============================================================================
*-=====       USER TESTING                                                =====
*-=============================================================================
*! <%GTREE Example 2: User checking conditions using globals %>
*- minimal checking on lower- and upperbound:
%Data%        IO(TreeElements,Years)
%Correct%     IO(TreeElements,Years)<=Upper(Years) and IO(TreeElements,Years)>=Lower(Years)

%UserTest%
*The test generates the output:
*   set CorrectIO(TreeElements,Years)
*   set WrongIO(TreeElements,Years)
%userdisplay% CorrectIO, WrongIO;
*=============================================================================
*- you can use %x% instead of IO(TreeElements,Years), this makes the condition easier to read
%Data%        IO(TreeElements,Years)
%Correct%     %x%<=Upper(Years) and %x%>=Lower(Years)

%UserTest%
%userdisplay% CorrectIO, WrongIO;
*=============================================================================
*- use the Tuple to check on only elements that are in the Tuple
set TestConditions2(TreeElements)
/
Chapter3
/;

%Data%    IO(TreeElements,Years)
%Tuple%   TestConditions2(TreeElements)
%Correct% %x%<=500 and %x%>10
%OutName% DataCheck

%UserTest%
%userdisplay% DataCheckCorrectIO, DataCheckWrongIO;
*=============================================================================
*- you can specify a Correct conditions as well as a wrong condition
%Data%        IO(TreeElements,Years)
%Tuple%       TestConditions2(TreeElements)
%Correct%     IO(TreeElements,Years)<=500 and IO(TreeElements,Years)>10
%Wrong%       IO(TreeElements,Years)<10

%UserTest%
%userdisplay% CorrectIO, WrongIO;

set NotChecked(TreeElements,Years);
NotChecked(TreeElements,Years)$(not CorrectIO(TreeElements,Years) and not WrongIO(TreeElements,Years))=yes
%userdisplay% ""These elements are not checked!""
%userdisplay% NotChecked;
*=============================================================================
*- Otherwise means that Wrong is all that isn't Correct (i.e. this is the default)
%Data%        IO(TreeElements,Years)
%Tuple%       TestConditions2(TreeElements)
%Correct%     IO(TreeElements,Years)<=500 and IO(TreeElements,Years)>10
%Wrong%       Otherwise

%UserTest%
%userdisplay% CorrectIO, WrongIO;
*=============================================================================
*- using Tuple All means that we perform checking on all elements of the data (All is default for Tuple)
%Data%    IO(TreeElements,Years)
%Tuple%   All
%Correct% IO(TreeElements,Years)<=Upper(Years) and IO(TreeElements,Years)>=Lower(Years)

%UserTest%
%userdisplay% CorrectIO, WrongIO;
*=============================================================================
*- changing the output names of the sets is possible
%Data%        IO(TreeElements,Years)
%Correct%     %x%<=Upper(Years) and %x%>=Lower(Years)
%OutName%     Bounds_
%CorrectName% Correct_
%WrongName%   Wrong_

%UserTest%
%userdisplay% Bounds_Correct_IO, Bounds_Wrong_IO;
*=============================================================================
*- you can use conditionals in the %Correct% and %Wrong%
%Data%        IO(TreeElements,Years)
%Correct%     (%x%<=Upper(Years) and %x%>=Lower(Years))$TestConditions(TreeElements,Years)
%UserTest%
%userdisplay% CorrectIO, WrongIO;
*=============================================================================


*!=============================================================================
*! ADDITIONAL EXAMPLES:
*!=============================================================================
*! <%GTREE Example 3: Normal Sign tests on the data %>
$setglobal DoSignTest $batinclude %SignTestDir%SignTest.gms
$setglobal DoUserTest $batinclude %SignTestDir%SignTest.gms User

%DoSignTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  StrictlyNegative
*            Data                   Tuple                               Condition
%userdisplay% StrictlyNegativeCorrectIO, StrictlyNegativeWrongIO;

%DoSignTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  Negative
*            Data                   Tuple                               Condition
%userdisplay% NegativeCorrectIO;

%DoSignTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  Zero
*            Data                   Tuple                               Condition
%userdisplay% ZeroCorrectIO;

%DoSignTest% IO(TreeElements,Years) IO(TreeElements,Years) StrictlyNegative
*            Data                   Tuple                  Condition
%userdisplay% StrictlyNegativeCorrectIO, StrictlyNegativeWrongIO;

*use a minus or a dot or All as the second argument to indicate that you want to check on all the data of argument 1
%DoSignTest% IO(TreeElements,Years) All     StrictlyPositive
*            Data                   Tuple   Condition
%userdisplay% StrictlyPositiveCorrectIO, StrictlyPositiveWrongIO;
*=============================================================================

*! <%GTREE Example 4: Sign testing using a Tuple with lower number of sets as the Data%>
%DoSignTest% IO(TreeElements,Years) TestConditions2(TreeElements)  Positive
*            Data                   Tuple                          Condition
%userdisplay% PositiveWrongIO;

%DoSignTest% IO(TreeElements,Years) TestConditions2(TreeElements)  StrictlyPositive
*            Data                   Tuple                          Condition
%userdisplay% StrictlyPositiveCorrectIO;
*=============================================================================

*! <%GTREE Example5: Demos of allowed tuple declaration%>
set TestConditions3(TreeElements,Years)
/
set.TreeElements    . (2000,2005,2010)
/;

set TestConditions4(TreeElements,Years)
/
Chapter2            . 2001
Chapter1            . set.Years
(Section1,Section2) . (2005*2010)
/;

set ChapterElements(TreeElements)/Chapter1,Chapter2,Chapter3/;

set CheckThis(TreeElements,Years)
/
set.ChapterElements    . (2000,2005,2010)
/;

%DoSignTest% IO(TreeElements,Years) CheckThis(TreeElements,Years)  Positive   CheckErrors
*            Data                   Tuple                          Condition  OutName
%userdisplay% CheckErrorsCorrectIO, CheckErrorsWrongIO;

%DoSignTest% IO(TreeElements,Years) CheckThis(TreeElements,Years)  Positive   Check    OK_         Error_
*            Data                   Tuple                          Condition  OutName  CorrectName WrongName
%userdisplay% CheckOK_IO,CheckError_IO;
*=============================================================================

*! <%GTREE Example 6: Subsetting data (ChapterElements instead of TreeElements)%>
%DoSignTest% IO(ChapterElements,Years) CheckThis(ChapterElements,Years)  Positive   SubSetErrors
*            Data                      Tuple                             Condition  OutName
%userdisplay% SubSetErrorsCorrectIO, SubSetErrorsWrongIO;

*! <%GTREE Example 7: User checking conditions%>
%DoUserTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  IO(TreeElements,Years)<=100  IO(TreeElements,Years)>100
*            Data                   Tuple                               Correct                      Wrong
%userdisplay% UserChecksCorrectIO, UserChecksWrongIO;

%DoUserTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  IO(TreeElements,Years)<=100  IO(TreeElements,Years)>100 User    InRange      OutofRange
*            Data                   Tuple                               Correct                      Wrong                      OutName CorrectName  WrongName
%userdisplay% UserInRangeIO, UserOutofRangeIO;


%DoUserTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  "IO(TreeElements,Years)<=100 and IO(TreeElements,Years)>10"  "IO(TreeElements,Years)>100 or IO(TreeElements,Years)<=10"
*            Data                   Tuple                               Correct                                                      Wrong
%userdisplay% UserChecksCorrectIO, UserChecksWrongIO;

%DoUserTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  "IO(TreeElements,Years)<=100 and IO(TreeElements,Years)>10"  "not (IO(TreeElements,Years)<=100 and IO(TreeElements,Years)>10)"
*            Data                   Tuple                               Correct                                                      Wrong
%userdisplay% UserChecksCorrectIO, UserChecksWrongIO;

%DoUserTest% IO(TreeElements,Years) TestConditions(TreeElements,Years)  "IO(TreeElements,Years)<=100 and IO(TreeElements,Years)>10"  Otherwise
*            Data                   Tuple                               Correct                                                      Wrong
%userdisplay% UserChecksCorrectIO, UserChecksWrongIO;

%DoUserTest% IO(TreeElements,Years) TestConditions2(TreeElements)  "IO(TreeElements,Years)<=500 and IO(TreeElements,Years)>10"  "IO(TreeElements,Years)>500 or IO(TreeElements,Years)<=10"
*            Data                   Tuple                          Correct                                                      Wrong
%userdisplay% UserChecksCorrectIO, UserChecksWrongIO;

%DoUserTest% IO(TreeElements,Years) TestConditions2(TreeElements)  "%x% <= 500 and %x% > 10"  Otherwise
*            Data                   Tuple                          Correct                Wrong
*NB using %x% instead of IO(TreeElements,Years) makes "Correct" and "Wrong" more readable/shorter
%userdisplay% UserChecksCorrectIO, UserChecksWrongIO;
*=============================================================================

*! <%GTREE Example 8: Set Union and Set Intersection%>
set Combined(TreeElements,Years);
Combined(TreeElements,Years)=TestConditions4(TreeElements,Years) or CheckThis(TreeElements,Years);
%userdisplay% Combined;

set InBoth(TreeElements,Years);
InBoth(TreeElements,Years)=TestConditions4(TreeElements,Years) and CheckThis(TreeElements,Years);
%userdisplay% InBoth;
*============================   End Of File   ================================